"use strict";
//
// Copyright 2023 Signal Messenger, LLC.
// SPDX-License-Identifier: AGPL-3.0-only
//
Object.defineProperty(exports, "__esModule", { value: true });
exports.chunkSizeInBytes = exports.ValidatingWritable = exports.DigestingWritable = exports.inferChunkSize = exports.everyNthByte = void 0;
const Native = require("../Native");
const stream = require("stream");
const Errors_1 = require("./Errors");
function everyNthByte(n) {
    return { kind: 'everyN', n: n };
}
exports.everyNthByte = everyNthByte;
function inferChunkSize(dataSize) {
    return { kind: 'chunksOf', dataSize: dataSize };
}
exports.inferChunkSize = inferChunkSize;
class DigestingWritable extends stream.Writable {
    constructor(key, sizeChoice) {
        super();
        this._digests = [];
        this._nativeHandle = Native.IncrementalMac_Initialize(key, chunkSizeInBytes(sizeChoice));
    }
    getFinalDigest() {
        return Buffer.concat(this._digests);
    }
    _write(
    // eslint-disable-next-line @typescript-eslint/no-explicit-any, @typescript-eslint/explicit-module-boundary-types
    chunk, _encoding, callback) {
        // eslint-disable-next-line @typescript-eslint/no-unsafe-argument
        const buffer = Buffer.from(chunk, 'binary');
        const next_digest = Native.IncrementalMac_Update(this, buffer, 0, buffer.length);
        if (next_digest.length != 0) {
            this._digests.push(next_digest);
        }
        callback();
    }
    _final(callback) {
        this._digests.push(Native.IncrementalMac_Finalize(this));
        callback();
    }
}
exports.DigestingWritable = DigestingWritable;
class ValidatingWritable extends stream.Writable {
    constructor(key, sizeChoice, digest) {
        super();
        this._validatedBytes = 0;
        this._nativeHandle = Native.ValidatingMac_Initialize(key, chunkSizeInBytes(sizeChoice), digest);
    }
    validatedSize() {
        return this._validatedBytes;
    }
    _write(
    // eslint-disable-next-line @typescript-eslint/no-explicit-any, @typescript-eslint/explicit-module-boundary-types
    chunk, _encoding, callback) {
        // eslint-disable-next-line @typescript-eslint/no-unsafe-argument
        const buffer = Buffer.from(chunk, 'binary');
        const validBytes = Native.ValidatingMac_Update(this, buffer, 0, buffer.length);
        if (validBytes >= 0) {
            this._validatedBytes += validBytes;
            callback();
        }
        else {
            callback(makeVerificationError('Corrupted input data'));
        }
    }
    _final(callback) {
        const validBytes = Native.ValidatingMac_Finalize(this);
        if (validBytes >= 0) {
            this._validatedBytes += validBytes;
            callback();
        }
        else {
            callback(makeVerificationError('Corrupted input data (finalize)'));
        }
    }
}
exports.ValidatingWritable = ValidatingWritable;
function chunkSizeInBytes(sizeChoice) {
    switch (sizeChoice.kind) {
        case 'everyN':
            return sizeChoice.n;
            break;
        case 'chunksOf':
            return Native.IncrementalMac_CalculateChunkSize(sizeChoice.dataSize);
            break;
    }
}
exports.chunkSizeInBytes = chunkSizeInBytes;
function makeVerificationError(message) {
    return new Errors_1.LibSignalErrorBase(message, 'VerificationFailed', 'incremental_mac');
}
//# sourceMappingURL=incremental_mac.js.map